/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

"use strict";

let original_Functions = new WeakMap();

Components.utils.import("resource://gre/modules/Services.jsm");


function install_Hooks(){
	let window_Object=this;

	let original_whereToOpenLink = window_Object.whereToOpenLink;
	window_Object.whereToOpenLink = function(e, ignoreButton, ignoreAlt){
		let normal_Result = original_whereToOpenLink.apply(window_Object, arguments);

		// It's fairly questionable how one should expect various application buttons (like the back, forward, and
		// reload buttons) to work when using a location other than the current window or tab and unsurprisingly they
		// usually don't work too well when you try it. For those buttons we'll just use the normal behavior so check
		// the event type value to see if it is "command" (which should indicate that it is an application button that
		// was clicked) and return the result from the original_whereToOpenLink function.
		if(e.type == "command")
			return normal_Result;

		if(!e)
			return normal_Result;

		if(e.metaKey){
			let modifier_Action=Services.prefs.getCharPref("extensions.Click_Modifiers.Meta_Click_Action");
			switch(modifier_Action){
				case "default":
					return normal_Result;

				case "tab":
					return (e.shiftKey ^ Services.prefs.getBoolPref("browser.tabs.loadInBackground")) ?
						"tabshifted" : "tab";

				case "backgroundtab":
					return (e.shiftKey ^ Services.prefs.getBoolPref("browser.tabs.loadInBackground")) ?
						"tab" : "tabshifted";

				default:
					return modifier_Action;
			}
		}
	
		if(e.ctrlKey){
			let modifier_Action=Services.prefs.getCharPref("extensions.Click_Modifiers.Control_Click_Action");
			switch(modifier_Action){
				case "default":
					return normal_Result;

				case "tab":
					return (e.shiftKey ^ Services.prefs.getBoolPref("browser.tabs.loadInBackground")) ?
						"tabshifted" : "tab";

				case "backgroundtab":
					return (e.shiftKey ^ Services.prefs.getBoolPref("browser.tabs.loadInBackground")) ?
						"tab" : "tabshifted";

				default:
					return modifier_Action;
			}
		}
	
		if((e.button==1) && !ignoreButton){
			let modifier_Action=Services.prefs.getCharPref("extensions.Click_Modifiers.Middle_Click_Action");
			switch(modifier_Action){
				case "default":
					return normal_Result;

				case "tab":
					return (e.shiftKey ^ Services.prefs.getBoolPref("browser.tabs.loadInBackground")) ?
						"tabshifted" : "tab";

				case "backgroundtab":
					return (e.shiftKey ^ Services.prefs.getBoolPref("browser.tabs.loadInBackground")) ?
						"tab" : "tabshifted";

				default:
					return modifier_Action;
			}
		}
	
		if(e.altKey && !ignoreAlt){
			let modifier_Action=Services.prefs.getCharPref("extensions.Click_Modifiers.Alt_Click_Action");
			switch(modifier_Action){
				case "default":
					return normal_Result;

				case "tab":
					return (e.shiftKey ^ Services.prefs.getBoolPref("browser.tabs.loadInBackground")) ?
						"tabshifted" : "tab";

				case "backgroundtab":
					return (e.shiftKey ^ Services.prefs.getBoolPref("browser.tabs.loadInBackground")) ?
						"tab" : "tabshifted";

				default:
					return modifier_Action;
			}
		}
	
		if(e.shiftKey){
			let modifier_Action=Services.prefs.getCharPref("extensions.Click_Modifiers.Shift_Click_Action");
			switch(modifier_Action){
				case "default":
					return normal_Result;

				case "tab":
					return Services.prefs.getBoolPref("browser.tabs.loadInBackground") ? "tabshifted" : "tab";

				case "backgroundtab":
					return Services.prefs.getBoolPref("browser.tabs.loadInBackground") ? "tab" : "tabshifted";

				default:
					return modifier_Action;
			}
		}

		return normal_Result;
	};


	// We need to modify the normal behavior of handleLinkClick() a little because it looks like neither it or
	// whatever usually calls it will do anything if a link is supposed to be opened in the "current" window/tab
	// and it was caused by a modified left click.
	let original_handleLinkClick = window_Object.handleLinkClick;
	window_Object.handleLinkClick = function(event, href, linkNode){
		let normal_Result = original_handleLinkClick.apply(window_Object, arguments);
	
		// Return normal_Result from original_handleLinkClick if it already handled the click, it was a left click with
		// no modifiers keys used, or it was a right click.
		if(normal_Result
		  || (event.button==0 && !(event.shiftKey || event.ctrlKey || event.metaKey || event.altKey))
		  || (event.button==2)){
			return normal_Result;
		}else{
			let doc = event.target.ownerDocument;
			window_Object.urlSecurityCheck(href, doc.nodePrincipal);
			window_Object.openLinkIn(href, window_Object.whereToOpenLink(event), { referrerURI: doc.documentURIObject,
			  charset: doc.characterSet });
			event.preventDefault();
			return true;
		}
	};

	// Save the original functions so we can unhook these functions later.
	original_Functions.set(window_Object, {	"whereToOpenLink": original_whereToOpenLink,
											"handleLinkClick": original_handleLinkClick}
	);
}


function remove_Hooks(){
	this.whereToOpenLink = original_Functions.get(this)["whereToOpenLink"];
	this.handleLinkClick = original_Functions.get(this)["handleLinkClick"];
	original_Functions.delete(this);
}


function ensure_Preferences_Are_Compatible_With(preference_To_Ensure_Compatibility_With){
	let click_Modifiers_Preferences=Services.prefs.getBranch("extensions.Click_Modifiers.");

	switch(preference_To_Ensure_Compatibility_With){
		case "extensions.Click_Modifiers.Alt_Click_Action":{
			let browser_Preferences=Services.prefs.getBranch("browser.");
			switch(click_Modifiers_Preferences.getCharPref("Alt_Click_Action")){
				case "default":
					if(browser_Preferences.prefHasUserValue("altClickSave"))
						browser_Preferences.clearUserPref("altClickSave");
					break;
		
				case "save":
					if(!browser_Preferences.getBoolPref("altClickSave"))
						browser_Preferences.setBoolPref("altClickSave", true);
					break;
		
				default:
					browser_Preferences.setBoolPref("altClickSave", false);
					break;
			}
			break;
		}

		case "browser.altClickSave":{
			// altClickSave		Is Default		Existing			-->	New
			//					Setting			Alt_Click_Action	-->	Alt_Click_Action
			//--------------------------------------------------------------------------
			// True				True			default				-->	default	(No change)
			// 									save				-->	save	(No change)
			// 									Other				-->	default
			//--------------------------------------------------------------------------
			// True				False			default				-->	save
			// 									save				-->	save	(No change)
			// 									Other				-->	save
			//--------------------------------------------------------------------------
			// False			True			default				-->	default	(No change)
			// 									save				-->	default
			// 									Other				-->	Other	(No change)
			//--------------------------------------------------------------------------
			// False			False			default				-->	default	(No change)
			// 									save				-->	default
			// 									Other				-->	Other	(No change)
			let browser_Preferences=Services.prefs.getBranch("browser.");
			if(browser_Preferences.getBoolPref("altClickSave")){
				if(browser_Preferences.prefHasUserValue("altClickSave")){
					if(click_Modifiers_Preferences.getCharPref("Alt_Click_Action")!="save")
						click_Modifiers_Preferences.setCharPref("Alt_Click_Action", "save");
				}else{
					switch(click_Modifiers_Preferences.getCharPref("Alt_Click_Action")){
						case "default":
						case "save":
							break;

						default:
							click_Modifiers_Preferences.setCharPref("Alt_Click_Action", "default");
							break;
					}
				}
			}else{
				if(click_Modifiers_Preferences.getCharPref("Alt_Click_Action")=="save")
					click_Modifiers_Preferences.setCharPref("Alt_Click_Action", "default");
			}
			break;
		}

		case "extensions.Click_Modifiers.Middle_Click_Action":{
			let opentabfor_Preferences=Services.prefs.getBranch("browser.tabs.opentabfor.");
			switch(click_Modifiers_Preferences.getCharPref("Middle_Click_Action")){
				case "default":
					if(opentabfor_Preferences.prefHasUserValue("middleclick"))
						opentabfor_Preferences.clearUserPref("middleclick");
					break;
		
				case "tab":
					if(!opentabfor_Preferences.getBoolPref("middleclick"))
						opentabfor_Preferences.setBoolPref("middleclick", true);
					break;
		
				default:
					opentabfor_Preferences.setBoolPref("middleclick", false);
					break;
			}
			break;
		}

		case "browser.tabs.opentabfor.middleclick":{
			// opentabfor.		Is Default		Existing			-->	New
			// middleclick		Setting			Middle_Click_Action	-->	Middle_Click_Action
			//--------------------------------------------------------------------------
			// True				True			default				-->	default	(No change)
			// 									tab					-->	tab		(No change)
			// 									Other				-->	default
			//--------------------------------------------------------------------------
			// True				False			default				-->	tab
			// 									tab					-->	tab		(No change)
			// 									Other				-->	tab
			//--------------------------------------------------------------------------
			// False			True			default				-->	default	(No change)
			// 									tab					-->	default
			// 									Other				-->	Other	(No change)
			//--------------------------------------------------------------------------
			// False			False			default				-->	default	(No change)
			// 									tab					-->	default
			// 									Other				-->	Other	(No change)
			let opentabfor_Preferences=Services.prefs.getBranch("browser.tabs.opentabfor.");
			if(opentabfor_Preferences.getBoolPref("middleclick")){
				if(opentabfor_Preferences.prefHasUserValue("middleclick")){
					if(click_Modifiers_Preferences.getCharPref("Middle_Click_Action")!="tab")
						click_Modifiers_Preferences.setCharPref("Middle_Click_Action", "tab");
				}else{
					switch(click_Modifiers_Preferences.getCharPref("Middle_Click_Action")){
						case "default":
						case "tab":
							break;

						default:
							click_Modifiers_Preferences.setCharPref("Middle_Click_Action", "default");
							break;
					}
				}
			}else{
				if(click_Modifiers_Preferences.getCharPref("Middle_Click_Action")=="tab")
					click_Modifiers_Preferences.setCharPref("Middle_Click_Action", "default");
			}
			break;
		}
	}
}


let preferences_Observer={
	observe: function(observed_Object, observed_Event, observed_Event_Data){
		if(observed_Event=="nsPref:changed")
			ensure_Preferences_Are_Compatible_With(observed_Event_Data);
	}
}


function window_Observer(observed_Window, observed_Event){
	switch(observed_Event){
		case "domwindowopened":
			// Unfortunately we can't just install the hooks right now because the functions which need to be hooked
			// aren't defined yet so what we do instead is add an event listener for the "load" event and then install
			// the hooks when the "load" event occurs.
			observed_Window.addEventListener("load", install_Hooks);
			break;

		case "domwindowclosed":
			observed_Window.removeEventListener("load", install_Hooks);
			remove_Hooks.call(observed_Window);
			break;
	}
}


function install(data, reason){}


function startup(data, reason){
	// Create our default preferences. This has to be done every startup due to bug 564675 
	// https://bugzilla.mozilla.org/show_bug.cgi?id=564675
	let click_Modifiers_Default_Preferences=Services.prefs.getDefaultBranch("extensions.Click_Modifiers.");
	click_Modifiers_Default_Preferences.setCharPref("Alt_Click_Action", "default");
	click_Modifiers_Default_Preferences.setCharPref("Control_Click_Action", "default");
	click_Modifiers_Default_Preferences.setCharPref("Meta_Click_Action", "default");
	click_Modifiers_Default_Preferences.setCharPref("Middle_Click_Action", "default");
	click_Modifiers_Default_Preferences.setCharPref("Shift_Click_Action", "default");

	// Check that all settings in extensions.Click_Modifiers are compatible with the current settings for
	// browser.altClickSave and browser.tabs.opentabfor.middleclick .
	ensure_Preferences_Are_Compatible_With("browser.altClickSave");
	ensure_Preferences_Are_Compatible_With("browser.tabs.opentabfor.middleclick");

	// Add observers for certain preferences so we can update other related preferences accordingly.
	Services.prefs.addObserver("extensions.Click_Modifiers.Alt_Click_Action", preferences_Observer, false);
	Services.prefs.addObserver("browser.altClickSave", preferences_Observer, false);
	Services.prefs.addObserver("extensions.Click_Modifiers.Middle_Click_Action", preferences_Observer, false);
	Services.prefs.addObserver("browser.tabs.opentabfor.middleclick", preferences_Observer, false);

	// Hook all the existing windows.
	let window_Enumerator = Services.wm.getEnumerator(null);
	while(window_Enumerator.hasMoreElements()){
		install_Hooks.call(window_Enumerator.getNext());
	}

	// Register to be notified when windows are opened or closed so we can hook newly opened windows and unhook windows
	// that are being closed.
	Services.ww.registerNotification(window_Observer);
}


function shutdown(data, reason){
	// Unregister for notifications about windows being opened/closed.
	Services.ww.unregisterNotification(window_Observer);

	// Unhook all the existing windows.
	let window_Enumerator = Services.wm.getEnumerator(null);
	while(window_Enumerator.hasMoreElements()) {
		remove_Hooks.call(window_Enumerator.getNext());
	}

	// Remove the observers we added for watching certain preferences.
	Services.prefs.removeObserver("extensions.Click_Modifiers.Alt_Click_Action", preferences_Observer);
	Services.prefs.removeObserver("browser.altClickSave", preferences_Observer);
	Services.prefs.removeObserver("extensions.Click_Modifiers.Middle_Click_Action", preferences_Observer);
	Services.prefs.removeObserver("browser.tabs.opentabfor.middleclick", preferences_Observer);
}


function uninstall(data, reason){
	// Delete our preferences if our extension is being uninstalled.
	if(reason == ADDON_UNINSTALL)
		Services.prefs.deleteBranch("extensions.Click_Modifiers");
}